/////////////////////////////////////////////////////////////////////////////////

// Original obtained from GlsSandbox.com
// Adapted, trivialy, for VGHD by TheEmu.
//
// Further adapted by TheEmu so that the Dome is circular, the load on the GPU
// is minimised and to increase flexibility by partialy parameterising it.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// The originals of these shaders did not take gl_FragCoord.w  into
// account so the iStripper scale attribute would  have  no  effect
// when used ina scene node that used one of them. I have therefore
// performed a global replace to substitute scaled_gl_FragCoord for
// gl_FragCoord and declare it here. TheEmu 2016/12/15

#define scaled_gl_FragCoord vec4(gl_FragCoord.xyz*gl_FragCoord.w,1.0)

// Use defines here rather than edit the body of the code.

#define time u_Elapsed
#define resolution u_WindowSize
#define mouse vec2(0.0,0.0)
#define surfacePosition vec2(2.0*scaled_gl_FragCoord.xy/resolution-1.0)

/////////////////////////////////////////////////////////////////////////////////

// Shader parameters.  The load on the GPU depends on NUM_STEPS,  so if the scene
// is jerky then lower it.  However,  changing  NUM_STEPS also affects the degree
// of cloudiness, so CLOUDY_MAX and CLOUDY_MIN should also be modified along with
// NUM_STEPS if the overall look of the scene is to be maintained.  The values of
// two cloudiness control parameters should be small,  between 1.0 and -1.0,  and
// control both the fraction of sky covered by clouds and their darkness,  though
// really dark clouds need large values for NUM_STEPS.  The  GPU load can also be 
// reduced by reducing the size of the dome using the SIZE factor.

#define NUM_STEPS 50 // was 70 in original version.
#define SIZE 1.00    // Size factor for dome
#define SPEED_1 0.50 // Overall speed factor
#define SPEED_2 0.27 // Additional speed factor for cloudiness

#define CLOUDY_MAX  0.10 // Maximum cloudiness
#define CLOUDY_MIN -0.10 // Minimum cloudiness

/////////////////////////////////////////////////////////////////////////////////

// Cloud dome. By David Hoskins,2014.
// https://www.shadertoy.com/view/4tf3RM#
// License Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.

// Renders half a sphere map for procedural sky rendering.
// Use this once to create the sky dome. Then use the IntoCartesian function in your sky rendering routine
// to find the UV coordinate in the texture in the same way you would use a cube map, with a ray direction.
// It uses the FLATTEN define to stop the mapping looking pixelated at the top.
// \/\/ - set to .08 for a useful flatten value for a dome texture, or .5 to see the clouds up close.

#define FLATTEN 0.4

#ifdef GL_ES
precision mediump float;
#endif

// uniform float time;      // Replaced for VGHD by defines above. TheEmu.
// uniform vec2 resolution; // Replaced for VGHD by defines above. TheEmu.

vec3 sunLight = normalize( vec3( 0.35, 0.22, 0.3 ) );
const vec3 sunColour = vec3(1.0, .86, .7);

#define cloudLower 2000.0
#define cloudUpper 3800.0

//#define TEXTURE_NOISE

float gTime  = 0.0;
float cloudy = 0.0;

//--------------------------------------------------------------------------

float Hash(vec2 p)
{
   p  = fract(p * vec2(5.3983, 5.4427));
   p += dot(p.yx, p.xy + vec2(21.5351, 14.3137));
   return fract(p.x * p.y * 95.4337);
}

//--------------------------------------------------------------------------

float Hash(vec3 p)
{
   p  = fract(p * vec3(5.3983, 5.4472, 6.9371));
   p += dot(p.yzx, p.xyz + vec3(21.5351, 14.3137, 15.3219));
   return fract(p.x * p.y * p.z * 95.4337);
}

//--------------------------------------------------------------------------
#ifdef TEXTURE_NOISE
float Noise( in vec3 x )
{
   vec3 p = floor(x);
   vec3 f = fract(x);
   f = f*f*(3.0-2.0*f);

   vec2 uv = (p.xy+vec2(37.0,17.0)*p.z) + f.xy;
   vec2 rg = texture2D( iChannel0, (uv+ 0.5)/256.0, -100.0 ).yx;
   return mix( rg.x, rg.y, f.z );
}
#else
//--------------------------------------------------------------------------
float Noise(in vec3 p)
{
   vec3 i = floor(p);
   vec3 f = fract(p);
   f *= f * (3.0-2.0*f);

    return mix(
      mix(mix(Hash(i + vec3(0.,0.,0.)), Hash(i + vec3(1.,0.,0.)),f.x),
          mix(Hash(i + vec3(0.,1.,0.)), Hash(i + vec3(1.,1.,0.)),f.x),
         f.y),
      mix(mix(Hash(i + vec3(0.,0.,1.)), Hash(i + vec3(1.,0.,1.)),f.x),
          mix(Hash(i + vec3(0.,1.,1.)), Hash(i + vec3(1.,1.,1.)),f.x),
         f.y),
      f.z);
}
#endif

//--------------------------------------------------------------------------

float FBM( vec3 p )
{
   p *= .5;
    float f;
    f  = 0.5000   * Noise(p); p = p * 3.02;
    f += 0.2500   * Noise(p); p = p * 3.03;
    f += 0.1250   * Noise(p); p = p * 3.01;
    f += 0.0625   * Noise(p); p = p * 3.03;
    f += 0.03125  * Noise(p); p = p * 3.02;
    f += 0.015625 * Noise(p);
    return f;
}

//--------------------------------------------------------------------------

float Map(vec3 p)
{
   float h = FBM(p);
   return h-cloudy-.42;
}

//--------------------------------------------------------------------------

// Grab all sky information for a given ray from camera

vec3 GetSky(in vec3 pos,in vec3 rd)
{
   float sunAmount = max( dot( rd, sunLight), 0.0 );
   // Do the blue and sun...
   vec3  sky = mix(vec3(.1, .1, .4), vec3(.1, .45, .7), 1.0-pow(abs(rd.y), .5));
   sky = sky + sunColour * min(pow(sunAmount, 1500.0) * 2.0, 1.0);
   sky = sky + sunColour * min(pow(sunAmount, 10.0) * .75, 1.0);

   // Find the start and end of the cloud layer...
   float beg = ((cloudLower-pos.y)/rd.y);
   float end = ((cloudUpper-pos.y)/rd.y);
   // Start position...
   vec3 p = vec3(pos.x + rd.x * beg, cloudLower, pos.z + rd.z * beg);

   // Trace clouds through that layer...
   float d = 0.0;
   float add = (end-beg) / float(NUM_STEPS);
   vec4 sum = vec4(0.1, .1, .1, 0.0);
   // Horizon fog is just thicker clouds...
   vec4 col = vec4(0.0, 0.0, 0.0, pow(1.0-rd.y,30.) * .2);
   for (int i = 0; i < NUM_STEPS; i++)
   {
      if (sum.a >= 1.0) continue;
      vec3 pos = p + rd * d;
      float h = Map(pos * .001);
      col.a += max(-h, 0.0) * .10;
      col.rgb = mix(vec3((pos.y-cloudLower)/((cloudUpper-cloudLower))) * col.a, sunColour, max(.3-col.a, 0.0) * .04);
      sum = sum + col*(1.0 - sum.a);
      d += add;
   }
   sum.xyz += min((1.-sum.a) * pow(sunAmount, 3.0), 1.0);
   sky = mix(sky, sum.xyz, sum.a);

   return clamp(sky, 0.0, 1.0);
}

//--------------------------------------------------------------------------

vec3 CameraPath( float t )
{
    return vec3(4000.0 * sin(.16*t), 0.0, 4000.0 * cos(.155*t) );
}

//--------------------------------------------------------------------------

vec3 IntoSphere(vec2 uv)
{
   vec3 dir;

   dir.x = uv.x;
   dir.z = uv.y;
   dir.y = sqrt(1.0 - dir.x*dir.x - dir.z*dir.z) * FLATTEN;

   return dir;

}

//--------------------------------------------------------------------------

void main(void)
{
   gTime = time*SPEED_1;

   cloudy = cos(gTime*SPEED_2);
   cloudy = cloudy * ( CLOUDY_MAX - CLOUDY_MIN ) + CLOUDY_MIN;

   vec2 uv = 2.0*scaled_gl_FragCoord.xy/resolution.xy - 1.0;

   float q = resolution.y / resolution.x;

   if ( q > 1.0 ) uv.y = uv.y / q;
   if ( q < 1.0 ) uv.x = uv.x / q;

   if ( length(uv) > SIZE ) 
    { gl_FragColor=vec4(0.0,0.0,0.0, 1.0);
    }
   else
    {
      vec3 dir = IntoSphere(uv/SIZE);
      vec3 col = GetSky(CameraPath(gTime), dir);
      // Don't gamma too much to keep the moody look...
      col = pow(col, vec3(.4));
      gl_FragColor=vec4(col, 1.0);
    }

   gl_FragColor *= gl_Color; // TheEmu 2016/12/15
}

//--------------------------------------------------------------------------
